// Copyright 2017 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "content/browser/blob_storage/blob_registry_wrapper.h"

#include "content/browser/blob_storage/chrome_blob_storage_context.h"
#include "content/browser/child_process_security_policy_impl.h"
#include "content/public/common/content_features.h"
#include "storage/browser/blob/blob_registry_impl.h"
#include "storage/browser/fileapi/file_system_context.h"

namespace content {

namespace {

class BindingDelegate : public storage::BlobRegistryImpl::Delegate {
 public:
  explicit BindingDelegate(int process_id) : process_id_(process_id) {}
  ~BindingDelegate() override {}

  bool CanReadFile(const base::FilePath& file) override {
    ChildProcessSecurityPolicyImpl* security_policy =
        ChildProcessSecurityPolicyImpl::GetInstance();
    return security_policy->CanReadFile(process_id_, file);
  }
  bool CanReadFileSystemFile(const storage::FileSystemURL& url) override {
    ChildProcessSecurityPolicyImpl* security_policy =
        ChildProcessSecurityPolicyImpl::GetInstance();
    return security_policy->CanReadFileSystemFile(process_id_, url);
  }

 private:
  const int process_id_;
};

}  // namespace

// static
scoped_refptr<BlobRegistryWrapper> BlobRegistryWrapper::Create(
    scoped_refptr<ChromeBlobStorageContext> blob_storage_context,
    scoped_refptr<storage::FileSystemContext> file_system_context) {
  scoped_refptr<BlobRegistryWrapper> result(new BlobRegistryWrapper());
  BrowserThread::PostTask(
      BrowserThread::IO, FROM_HERE,
      base::BindOnce(&BlobRegistryWrapper::InitializeOnIOThread, result,
                     std::move(blob_storage_context),
                     std::move(file_system_context)));
  return result;
}

BlobRegistryWrapper::BlobRegistryWrapper() {
  DCHECK(base::FeatureList::IsEnabled(features::kMojoBlobs));
}

void BlobRegistryWrapper::Bind(
    int process_id,
    storage::mojom::BlobRegistryRequest request) {
  DCHECK_CURRENTLY_ON(BrowserThread::IO);
  blob_registry_->Bind(std::move(request),
                       base::MakeUnique<BindingDelegate>(process_id));
}

BlobRegistryWrapper::~BlobRegistryWrapper() {}

void BlobRegistryWrapper::InitializeOnIOThread(
    scoped_refptr<ChromeBlobStorageContext> blob_storage_context,
    scoped_refptr<storage::FileSystemContext> file_system_context) {
  DCHECK_CURRENTLY_ON(BrowserThread::IO);
  blob_registry_ = base::MakeUnique<storage::BlobRegistryImpl>(
      blob_storage_context->context(), std::move(file_system_context));
}

}  // namespace content
